=begin
#-------------------------------------------------------------------------------------------------------------------------------------------------
#*************************************************************************************************
# Designed Dec. 2008 by Fredo6

# Permission to use this software for any purpose and without fee is hereby granted
# Distribution of this software for commercial purpose is subject to:
#  - the expressed, written consent of the author
#  - the inclusion of the present copyright notice in all copies.

# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------
# Name			:  Lib6Web.rb
# Original Date	:  10 Dec 2008 - version 3.0
# Type			:  Script library part of the LibFredo6 shared libraries
# Description	:  A utility library to assist web dialog design.
#-------------------------------------------------------------------------------------------------------------------------------------------------
#*************************************************************************************************
=end

module Traductor

T_HTML_DOC_TYPE = %q(<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN" "http://www.w3.org/TR/html4/strict.dtd">)
T_HTML_UTF_TYPE = %q(<META http-equiv='Content-Type' Content='text/html; charset=UTF-8'> )
T_HTML_SCRIPT_TYPE = %q(<META http-equiv='Content-Script-Type' content='text/javascript'>)

#--------------------------------------------------------------------------------------------------------------
# Class VTYPE: Manage validation types
#--------------------------------------------------------------------------------------------------------------			 

VTYPE_INVALID = 1
VTYPE_CONSTRAINT = 2

class VTYPE

attr_reader :strdef, :type, :vmin, :vmax, :message, :extra, :error

#Create a vtype instances from an encoded string
def initialize(strdef, extra, message=nil)
	@strdef = strdef
	@message = message
	@type = nil
	@error = 0
	@extra = extra
	self.parse()
end

#Parse the Vtype definition from a string specification
def parse
	l = @strdef.split ':'
	type = l[0]
	spec = (l.length > 1) ? l[1] : ""
	
	@type = type.strip.upcase
	@vmin = nil
	@vmax = nil
	@pattern = nil
	
	case @type
	when 'B', 'H', '/', 'M', 'O'
	when 'I', 'L', 'F'
		parse_minmax(spec)
	when 'S', 'D', 'K'
		@pattern = spec.strip
	else
		@type = 'S'
	end
end

#Parse the min max value for numbers
def parse_minmax(spec)
	return unless spec
	spec = spec.strip
	
	case spec
	when /<=(.*)>/, /<=(.*)/
		@vmax = '<=' + $1.strip
	when /<(.*)>/, /<(.*)/
		@vmax = '<' + $1.strip
	end
	
	case spec
	when />=(.*)</, />=(.*)/
		@vmin = '>=' + $1.strip
	when />(.*)</, />(.*)/
		@vmin = '>' + $1.strip
	end
end

#Interpret a string as a boolean
def VTYPE.parse_as_bool(svalue)
	case svalue
	when /true/i, /yes/i, true
		return true
	when /false/i, /no/i, false
		return false
	else
		return nil
	end	
end

#Return the real typed value from an output value
def real_value(svalue)
	svalue = svalue.to_s.strip
	val = nil
	begin
		case @type
		when 'I'
			val = Traductor.string_to_integer_formula svalue
		when 'F'
			val = Traductor.string_to_float_formula svalue
		when 'L'
			val = Traductor.string_to_float_formula svalue
		when 'B'
			val = VTYPE.parse_as_bool(svalue)
			return VTYPE.parse_as_bool(svalue)
		when 'H'
			val = KeyList.to_key @extra, svalue
		when 'M', 'O'
			val = svalue
		else
			val = svalue
		end	
	rescue
		return nil
	end
	val	
end

#Return the display value as a string (possibly translated) for a real value
def display_value(value)
	svalue = value.to_s.strip
	return "" if svalue == ""
	sval = svalue
	begin
		case @type
		when 'B'
			case svalue
			when /true/i, /yes/i, true
				return 'true'
			when /false/i, /no/i, false
				return 'false'
			else
				return ''
			end	
		when 'H'
			sval = KeyList.to_value @extra, svalue
		when 'M', 'O'
			sval = KeyList.to_value @extra, svalue
		else
			sval = svalue
		end	
	rescue
		return ''
	end
	sval	
end

def as_text
	text = ""
	case @type
	when 'I'
		text = "Integer"
	when 'F'
		text = "Float"
	when 'L'
		text = "Length"
	when 'B'
		text = "Boolean"
	when 'K'
		text = "Sketchup color"
	when 'D'
		text = "Directory"
	when 'S'
		text = "String"
	when 'H'
		text = "Single selection"
	when 'M'
		text = "Multiple selection"
	when 'O'
		text = "Multiple Ordered selection"
	else
		text = "Free"
	end
	
	#additional Contsraints
	case @type
	when 'I', 'L', 'F'
		text += " #{vmin} " if @vmin
		text += " #{vmax} " if @vmax
	when 'S'
		text += " pattern -> #{@pattern}" if @pattern
	end

	text
end

#Validate a value given in a string - Return value must be tested with (val == nil) to differentiate between real value and invalid value
def validate(svalue, error_level=0)
	#check if the value if valid, according to the type
	val = real_value svalue
	if val == nil
		@error = VTYPE_INVALID
		return nil
	end	
	
	#checking if the value respect the conditions
	status = true
	case @type
	when 'I', 'L', 'F'
		if @vmin
			begin
				status = status && eval(svalue + @vmin)
			rescue
			end
		end	
		if @vmax
			begin
				status = status && eval(svalue + @vmax)
			rescue
			end
		end	
	when 'S'
		if @pattern
			rpat = Regexp.new @pattern, Regexp::IGNORECASE
			status = rpat.match svalue
		end	
	when 'D'
	
	end
	
	#returning the value
	if status
		@error = 0
	else	
		@error = VTYPE_CONSTRAINT
		val = nil
	end	
	val
end

def compare(value1, value2)
	v1 = value1.to_s.upcase
	v2 = value2.to_s.upcase
	v1 <=> v2
end

def equal(value1, value2)
	v1 = value1.to_s.upcase
	v2 = value2.to_s.upcase
	v1 == v2
end

end	#class VTYPE

#--------------------------------------------------------------------------------------------------------------
# Class HTML: Manage an HTML Stream to power web dialog boxes
#--------------------------------------------------------------------------------------------------------------			 

T6[:T_Text_Default] = "Default:"
T6[:T_Text_Type] = "Type:"

class HTML

@@browser = nil
Traductor_HTML_Style = Struct.new("Traductor_HTML_Style", :classname, :parent, :computed, :lattr, :html) 

#Create an HTML stream object
def initialize
	@head = ""
	@body = ""
	@hstyles = {}
	include_standard_styles
end

def get_head
	@head
end

def get_body
	@body
end
	
def get_html_for_all_styles
	text = ""
	if @hstyles.length > 0
		text += "<style type='text/css'>"
		text += "* { margin: 0; padding: 2px; }" if RUN_ON_MAC
		@hstyles.each do |key, style|
			compute_style style
			text += ' ' + style.html + ' '
		end	
		text += "</style>"
	end
	text
end
	
#Create a Style structure identified by its classname
def create_style(classname, parent, *args)
	return unless classname && classname.strip != ""
	
	#Creating or identifying the style
	style = @hstyles[classname]
	unless style
		style = Traductor_HTML_Style.new
		style.computed = false
		style.classname = classname
		style.parent = parent
		style.html = ""
		style.lattr = []
		@hstyles[classname] = style
	end	
	
	#Calculating the attributes
	lattr = []
	args.each { |s| lattr += s.split(';') }
	lattr.each do |s|
		case s.strip
		when /\AB\Z/i		#bold --> 'B'
			sval = 'font-weight: bold'
		when /\AI\Z/i		#Italic --> 'I'
			sval = 'font-style: italic'
		when /\AK:(.*)/i	#SU color --> 'K:<color>'
			color = HTML.color($1) 
			sval = 'color: ' + color if color 
		when /\ABG:(.*)/i	#SU color --> 'BG:<color>'
			color = HTML.color($1) 
			sval = 'background-color: ' + color if color 
		when /\AF-SZ:\s*(\d*)/i, /\AF-SZ:\s*(\d+)pt/i	#Font size in pt --> 'F-sz:<size>'
			npx = ($1 == "") ? '1' : $1
			sval = "font-size: #{npx}pt" 
		when /\AF-SZ:\s*(\d*)px/i	#Font size in pixel --> 'F-SZ:<size>' 
			npx = ($1 == "") ? '1' : $1
			sval = "font-size: #{npx}px" 
		when /\ABD:\s*(.*)/i	#Border style' 
			sval = "border-style: #{$1}" 
		when /\ABD-SZ:\s*(\d*)/i, /\ABD-SZ:(\d*)px/i	#Border size in px --> 'Bd-sz:<size>'
			npx = ($1 == "") ? '1' : $1
			sval = "border-width: #{npx}pt" 
		when /\ABD-COL:\s*(.*)/i	#SU color --> 'K:<color>'
			color = HTML.color($1) 
			sval = 'border-color: ' + color if color 
		else
			sval = s.strip
		end	
		next if sval == ""
		style.lattr += [sval]
	end	
end

def include_standard_styles
	#style for custom tooltips
	create_style 'T_ToolTip', nil, 'visibility: hidden;', 'position: absolute;', 
				                   'top: 0;', 'left: 0;', 'z-index: 1000;', 'display: block',
								   'font: normal 8pt sans-serif;', 'padding: 3px;',
								   'border: solid 1px;', 'BG: yellow ;'

	#style for multi list
	create_style 'T_DivMultiList', nil, 
	             'height: 80px', 'overflow: auto', 'BD-SZ: 1',
				 'Bd: solid', 'Bd-col: LightGrey', 'cellspacing: 0', 'align: center', 
				 'F-SZ: 10', 'K:Black', 'font-weight: normal'

	#styles related to vScroll
	if HTML.browser =~ /6/ || RUN_ON_MAC
		create_style 'T_DivVScroll', nil, 'overflow: auto', 'margin-right: 15px'
		create_style 'T_DivVHeader', nil, 'margin-right: 15px'
	else	
		create_style 'T_DivVScroll', nil, 'overflow-y: scroll', 'overflow-x: hidden'
		create_style 'T_DivVHeader', nil
	end
end

#Calculate the final HTML String for the style
def compute_style(style)
	return if style.computed
	
	#Computing recursively the parent
	lattr = style.lattr
	sparent = (style.parent) ? @hstyles[style.parent] : nil
	style.computed = true
	unless sparent == nil || sparent.computed
		compute_style sparent
	end	
	lattr = sparent.lattr + lattr if sparent
	
	#Removing duplicate attributes and computing final HTML string
	return if lattr.length == 0	
	hkey = {}
	lattr.each { |s| hkey[$`.strip.upcase] = s if (s =~ /:/) }
	style.html = ".#{style.classname} { " + hkey.values.join(' ; ') + "}" if lattr.length > 0
end

def head_add(*args)
	args.each { |html| @head += html if html }
end	

def body_add(*args)
	args.each { |html| @body += html if html }
end	

#=========================================================
# Class methods to help building HTML flow
#=========================================================

#Return the browser type
def HTML.browser
	@@browser = SysInfo['BROWSER'] unless @@browser
	@@browser
end

#Return the offset for last column of table within a vertical scrolling DIV
def HTML.vscrolltable_extra(width)
	width += 15 unless (HTML.browser =~ /6/)
	"#{width}px"
end

#Compute an HTML color from a SU Color
def HTML.color(colorname)
	begin
		color = Sketchup::Color.new colorname
		s = ""
		s += sprintf "%02x", color.red
		s += sprintf "%02x", color.green
		s += sprintf "%02x", color.blue
		return '#' + s
	rescue
		return colorname
	end	
end

#Compute a constrating color, black or white
def HTML.contrast_color(colorname)
	begin
		color = Sketchup::Color.new colorname
		return ((color.red + color.green + color.blue) > 300) ? "#000000" : "#FFFFFF"
	rescue
		return "#000000"
	end	
end

#Format the event callbacks from a list of actions
def HTML.format_actions(lst_actions)
	lst_actions = [lst_actions] unless lst_actions.class == Array
	ls = lst_actions.uniq
	txt = ""
	ls.each do |action| 
		a = action.strip
		txt += " #{a}='Action(" + '"' + a + '"' + ", this)'" 
	end	
	txt
end

#Format a text element as a poragraph
def HTML.format_para(text, id="", classname="", extra_actions=[], tooltip="")
	attr = HTML.format_attr id, classname, [], extra_actions, tooltip
	text = text.gsub /[\n]/, "<br>"
	"<p #{attr}>" + text + '</p>'
end

#Format a text element as a span
def HTML.format_span(text, id="", classname="", extra_actions=[], tooltip="")
	attr = HTML.format_attr id, classname, [], extra_actions, tooltip
	text = text.gsub /[\n]/, "<br>"
	"<span #{attr}>" + text + '</span>'
end

#Format a text element as a div
def HTML.format_div(text, id="", classname="", extra_actions=[], tooltip="")
	attr = HTML.format_attr id, classname, [], extra_actions, tooltip
	text = text.gsub /[\n]/, "<br>"
	"<div #{attr}>" + text + '</div>'
end

#Format a Entry field
def HTML.format_input(text, nbchar, id="", classname="", extra_actions=[], tooltip="")
	attr = HTML.format_attr id, classname, ['onChange'], extra_actions, tooltip
	bchar = (nbchar) ? "maxlength='#{nbchar}' size='#{nbchar+1}'" : ""
	"<input type='text' #{bchar} value=\"#{text}\" #{attr}/>"
end

#Format a Push button
def HTML.format_button(text, id="", classname="", extra_actions=[], tooltip="")
	attr = HTML.format_attr id, classname, ['onClick'], extra_actions, tooltip
	"<input type='button' style='cursor:pointer' value=\"#{text}\" #{attr}/>"
end

#Format an image link
def HTML.format_imagelink(imgsrc, px, py, id="", classname="", extra_actions=[], tooltip="")
	attr = HTML.format_attr id, classname, ['onClick'], extra_actions, tooltip
	href = "style='cursor:pointer'"
	imgsrc = HTML.image_file imgsrc
	"<a #{href} #{attr}><img src='#{imgsrc}' height='#{py}' width='#{px}' border='0'/></a>"
end

#Format an image link
def HTML.format_textlink(text, id="", classname="", extra_actions=[], tooltip="")
	attr = HTML.format_attr id, classname, ['onClick'], extra_actions, tooltip
	href = "style='cursor:pointer ; text-decoration: underline'"
	"<a #{href} #{attr}>#{text}</a>"
end

#Format a Checkbox
def HTML.format_checkbox(bool, text, id="", classname="", extra_actions=[], tooltip="")
	attr = HTML.format_attr id, classname, ['onClick'], extra_actions, tooltip
	checked = (bool) ? "checked='checked'" : ""
	"<input type='checkbox' value=\"#{text}\" #{checked} #{attr}>#{text}</input>"
end

#Format a Combobox
def HTML.format_combobox(value, klist, id="", classname="", extra_actions=[], tooltip="", &extraproc)
	attr = HTML.format_attr id, classname, ['onChange'], extra_actions, tooltip
	txt = "<select #{attr}>"
	KeyList.each(klist) do |code, s|
		hextras = (extraproc) ? yield(code) : nil
		next if hextras && hextras['skip']
		exattr = (hextras && hextras['attr']) ? hextras['attr'] : "" 
		text = (hextras && hextras['text']) ? hextras['text'] : s 
		selected = ((code == value) || (code.upcase == value.upcase)) ? 'selected' : ''
		txt += "<option value='#{code}' #{exattr} #{selected}>#{text}</option>"
	end	
	txt += "</select>"
	txt
end

#Format a Combobox for Color picking
def HTML.format_SUcolorpicker(value, list, id="", classname="", extra_actions=[], tooltip="")
	attr = HTML.format_attr id, classname, ['onChange'], extra_actions, tooltip
	txt = "<select #{attr}>"
	list.each do |s|
		selected = (s.upcase == value.upcase) ? 'selected' : ''
		txcol = "style='color: #{HTML.contrast_color s} ; background-color: #{HTML.color s}'"	
		txt += "<option #{txcol} value='#{s}' #{selected}>#{s}</option>"
	end	
	txt += "</select>"
	txt
end

#Format a control area for a multi-selection non-ordered list
def HTML.format_multi_list(jsvalue, klist, id="", classname="", extra_actions=[], jsdefval=nil, hmax = nil)
	lsel = jsvalue.split(';;')
	ldef = jsdefval.split(';;')
	vlist = KeyList.values klist
	ylist = KeyList.keys klist
	
	#Creating the Div, enclosing the control
	if (vlist.length == 1)
		h = 20
	else
		hmax = 80 unless hmax
		hmax = 40 if hmax < 40
		h = vlist.length * 20
		h = hmax if h > hmax
		h = 40 if h < 40
	end	
	
	txt = ""
	txt += "<table width='99%' cellpadding='0px'><tr>"
	txt += "<td width='90%' align='left'>"
	
	txt += "<div width=99% style='height:#{h}px' class='T_DivMultiList'>"
	
	#Creating the value field
	txt += "<input type='hidden' value='#{jsvalue}' id='#{id}'/>"
	
	#Creating the list of options, with relevant status
	vlist.each_index do |i|
		id_option = id + "_Option____#{i}"
		checked = (lsel.include?(ylist[i])) ? "checked='checked'" : ""
		defval = (ldef.include?(ylist[i]))
		tdef = "title='#{T6[:T_Text_Default]} " + ((defval) ? 'true' : 'false') + "'"
		attr = HTML.format_attr id_option, classname, nil, nil, ""
		action = "onclick='multi_changed(\"#{id}\")'"
		txt += "<input type='checkbox' value='#{ylist[i]}' #{tdef} #{checked} #{action} #{attr}>"
		txt += "<span #{tdef}>#{vlist[i]}</span></input>"
		txt += "<br>"
	end	
	txt += "</div></td>"
	
	#Buttons clear and select all
	if vlist.length > 1
		txt += "<td width='10%' align='left' valign='top'>"
		imgall = HTML.image_file MYPLUGIN.picture_get("Button_Check.png")
		idall = "id='#{id}__All'"
		titall = "title='#{T6[:T_BUTTON_SelectAll]}'"
		actionall = "onclick='multi_select_all(\"#{id}\")'"

		imgclear = HTML.image_file MYPLUGIN.picture_get("Button_Clear.png")
		idclear = "id='#{id}__Clear'"
		titclear = "title='#{T6[:T_BUTTON_ClearAll]}'"
		actionclear = "onclick='multi_clear(\"#{id}\")'"
		href = "style='cursor:pointer'"
		space = "hspace='2' vspace='2' height='16' width='16' border='0'"
		txt += "<a #{href}><img src='#{imgall}' #{idall} #{titall} #{actionall} #{space}/></a><br>"
		txt += "<a #{href}><img src='#{imgclear}' #{idclear} #{titclear} #{actionclear} #{space}/></a>"
		txt += "</td>"
	end
	
	txt += "</tr></table>"

	txt
end

#Format a control area for a multi-selection ordered list
def HTML.format_ordered_list(jsvalue, klist, id="", classname="", extra_actions=[], jsdefval=nil, 
                             hmax = nil, colsel='lightcyan')
	lsel = jsvalue.split(';;')
	ldef = jsdefval.split(';;')
	vlist = KeyList.values klist
	ylist = KeyList.keys klist
	
	#Only one element in the list
	if (vlist.length == 1) 
		return HTML.format_multi_list(jsvalue, klist, id, classname, extra_actions, jsdefval, hmax)
	end
	
	#Creating the Div, enclosing the control
	hmax = 80 unless hmax
	hmax = 40 if hmax < 40
	h = vlist.length * 20
	h = hmax if h > hmax
	h = 40 if h < 40
	txt = ""
	txt += "<table width='99%' cellpadding='0px'><tr>"
	txt += "<td width='90%' align='left'>"
	
	txt += "<div width=99% style='height:#{h}px' class='T_DivMultiList'>"
	
	#Selection color
	colsel = 'lightcyan' unless colsel && colsel.strip != ""
	hcolsel = HTML.color colsel
	
	#Creating the value field
	txt += "<input type='hidden' value='#{jsvalue}' id='#{id}'/>"
	idsel = "#{id}_Selection____"
	idcol = "#{id}_Color____"
	txt += "<input type='hidden' value='' id='#{idsel}'/>"
	txt += "<input type='hidden' value='#{hcolsel}' id='#{idcol}'/>"
	
	#creating the correspondance list for ordering
	lsorted = []
	lsel.each { |v| j = ylist.index(v) ; lsorted.push j if j }
	ylist.each_index { |i| lsorted.push i unless lsorted.include?(i) }
	
	#Creating the table for options
	id_table = id + "_Table____"
	txt += "<table id='#{id_table}' cellspacing='0' cellpadding='0' width='90%'>"
	lsorted.each_with_index do |i, k|
		id_option = id + "_Option____#{i}"
		checked = (lsel.include?(ylist[i])) ? "checked='checked'" : ""
		defval = (ldef.include?(ylist[i]))
		tdef = "title='#{T6[:T_Text_Default]} " + ((defval) ? 'true' : 'false') + "'"
		attr = HTML.format_attr id_option, classname, nil, nil, ""
		action = "onclick='ordered_changed(\"#{id}\", \"#{i}\")'"
		trid = id + "_tr____#{i}"
		tdid = id + "_td____#{i}"
		tdaction = "onclick='ordered_highlight(\"#{id}\", \"#{i}\")'"
		txt += "<tr id='#{trid}' style='cursor:pointer'>"
		txt += "<td>"
		txt += "<input type='checkbox' style='cursor:default' "
		txt += "value='#{ylist[i]}' #{tdef} #{checked} #{action} #{attr}>"
		txt += "</td>"
		txt += "<td id='#{tdid}' #{tdaction} style='cursor:pointer'>"
		txt += "<span #{tdef}>#{vlist[i]}</span></input>"
		txt += "</td></tr>"
	end	
	txt += "</table>"
	txt += "</div></td>"
	txt += "<td width='10%' align='left' valign='top'>"

	imgall = HTML.image_file MYPLUGIN.picture_get("Button_Check.png")
	idall = "id='#{id}__All'"
	titall = "title='#{T6[:T_BUTTON_SelectAll]}'"
	actionall = "onclick='ordered_select_all(\"#{id}\")'"

	imgclear = HTML.image_file MYPLUGIN.picture_get("Button_Clear.png")
	idclear = "id='#{id}__Clear'"
	titclear = "title='#{T6[:T_BUTTON_ClearAll]}'"
	actionclear = "onclick='ordered_clear(\"#{id}\")'"

	imgup = HTML.image_file MYPLUGIN.picture_get("Button_Up.png")
	idup = "id='#{id}__Up'"
	titup = "title='#{T6[:T_BUTTON_Up]}'"
	actionup = "onclick='ordered_move_row (\"#{id}\", \"up\")'"

	imgdown = HTML.image_file MYPLUGIN.picture_get("Button_Down.png")
	iddown = "id='#{id}__Down'"
	titdown = "title='#{T6[:T_BUTTON_Down]}'"
	actiondown = "onclick='ordered_move_row (\"#{id}\", \"down\")'"
	
	href = "style='cursor:pointer'"
	space = "hspace='2' vspace='2' height='16' width='16' border='0'"
		txt += "<a #{href}><img src='#{imgup}' #{idup} #{titup} #{actionup} #{space}/></a><br>"
		txt += "<a #{href}><img src='#{imgdown}' #{iddown} #{titdown} #{actiondown} #{space}/></a>"
	if (vlist.length > 3)
		txt += "<br>"
		txt += "<a #{href}><img src='#{imgall}' #{idall} #{titall} #{actionall} #{space}/></a><br>"
		txt += "<a #{href}><img src='#{imgclear}' #{idclear} #{titclear} #{actionclear} #{space}/></a>"
	end	
	txt += "</td></tr></table>"

	txt
end

#Utility to format Id, classname and actions
def HTML.format_attr(id, classname, default_actions, extra_actions, tooltip)
	lst_actions = ((default_actions) ? default_actions : []) + ((extra_actions) ? extra_actions : [])
	tid = ( id && id != "") ? "id='#{id}'" : ""
	actions = HTML.format_actions lst_actions
	attr = (classname && classname != "") ? "class= '#{classname}'" : ""
	ttip = ""
	if (tooltip && tooltip != "")
		if tooltip =~ /\A_TT/i
			actions += " onmouseover='tooltip_show(\"#{tooltip}\");'"
			actions += " onmouseout='tooltip_hide(\"#{tooltip}\");'"
		else
			ttip = "title='#{tooltip}'"
		end	
	end	
	return "#{tid} #{ttip} #{attr} #{actions}"
end

def HTML.image_file(imgsrc)
	(RUN_ON_MAC && !(imgsrc =~ /file:\/\//i)) ? "file://" + imgsrc : imgsrc
end

end	#class HTML

#--------------------------------------------------------------------------------------------------------------
# Class Wdlg: Manage web dialogs
#--------------------------------------------------------------------------------------------------------------			 

class Wdlg

def initialize(title, hkey=nil, resizable=true, scrollable=false)
	@title = title
	@suwdlg = nil
	@scrollable = scrollable
	@resizable = resizable
	@xleft = 0
	@ytop = 0
	@width = 400
	@height = 300
	#@hkey = (hkey && !RUN_ON_MAC) ? hkey : ""
	@hkey = (hkey && !RUN_ON_MAC) ? hkey : nil
	@proc_close = nil
	@nb_callback = -1
	@asynchronous = (RUN_ON_MAC) ? true : false
end

def set_size(width, height, height_max=nil)
	@width = width
	@height = height
	@height_max = height_max
	@suwdlg.set_size @width, @height if @suwldg
end	

#Set the optional procedure to be called when the dialog box is closed
def set_on_close(&proc_cmd)
	@proc_close = proc_cmd
end

def set_position(xleft, ytop)
	@xleft = xleft
	@ytop = ytop
	@suwdlg.set_position @xleft, @ytop if @wldg	
end	

def visible?()
	(@suwdlg && @suwdlg.visible?) ? true : false
end

def bring_to_front()
	@suwdlg.bring_to_front if @suwdlg && @suwdlg.visible?
end

def set_background_color(color)
	@bgcolor = HTML.color color
end

def set_callback(hmethod)
	@hmethod = hmethod
end

def set_html_text(text_html)
	@text_html = text_html
	@suwdlg.set_html @text_html if @suwdlg
end

#specify the HTML structure to construct the page
def set_html(html)
	@text_html = assemble_html html
	@suwdlg.set_html @text_html if @suwdlg
end	
	
def create_dialog
	@suwdlg = UI::WebDialog.new @title, @scrollable, @hkey, @xleft, @ytop, @width, @height, @resizable
	@suwdlg.set_size @width, @height
	@suwdlg.set_background_color @bgcolor if @bgcolor
	@suwdlg.set_html @text_html if @text_html
	@suwdlg.set_on_close { j_onclose }
	@suwdlg.add_action_callback("Unique") { |d, p| self.j_dispatch(p) }
	@closing = false
end

def show(&proc)
	create_dialog unless @suwdlg
	if RUN_ON_MAC
		@suwdlg.navigation_buttons_enabled = false if SU_MAJOR_VERSION >= 7
		@suwdlg.show_modal {proc.call if proc} unless @suwdlg.visible?
		bring_to_front
	else	
		@suwdlg.show {proc.call if proc} unless @suwdlg.visible?
	end	
end

def show_modal(&proc)
	create_dialog unless @suwdlg
	@suwdlg.show_modal {proc.call if proc} unless @suwdlg.visible?
end

def close
	return unless @suwdlg
	@closing  = true
	@suwdlg.close if @suwdlg.visible?
end

#Set the initial focus for a web dialog box
def initial_focus(id, select=false)
	@inifocus = id
	@inisel = select
end

#Set the focus on a particular element
def put_focus(id, select=true)
	return unless id
	if select
		@suwdlg.execute_script "j6_put_focus ('#{id}', true) ;"
	else
		@suwdlg.execute_script "j6_put_focus ('#{id}', false) ;"
	end
end

#Get the element which has the focus
def get_focus()
	@suwdlg.execute_script "j6_get_focus() ;"
	@j_passback
end

def set_element_enabled(id, bflag=true)
	jscript_set_prop(id, 'disabled', !bflag)
end

#Set a value according to the type
def set_element_value(id, vtype, value, flagevent=false)
	return if value == nil
	type_elt = jscript_get_prop(id, 'type')
	if vtype.class == String
		stype = vtype
		vtype = nil
	elsif vtype.class == Traductor::VTYPE
		stype = vtype.type
	else
		stype = ''
		vtype = nil
	end
	
	case stype
	when 'B'
		prop = "checked"
		val = (value) ? true : false
		#return
	when 'I', 'F', 'L', 'K', 'S', 'D'
		prop = "value"
		val = value.to_s
	when 'H'
		prop = "value"
		#val = (vtype) ? KeyList.to_value(vtype.extra, value) : value
		val = (vtype) ? KeyList.to_key(vtype.extra, value) : value
	when 'M', 'O'
		prop = "value"
		val = (value.class == Array) ? value.join(';;') : value.to_s
	else
		prop = "innerHTML"
		val = value.to_s
	end	
	a = jscript_set_prop id, prop, val
	
	#Special treatment for custom controls
	case stype 
	when /O/i
		@suwdlg.execute_script "ordered_change(\"#{id}\")"
	when /M/i
		@suwdlg.execute_script "multi_change(\"#{id}\")"	
	end
	
	#notifying the event if needed
	@hmethod.call 'OnChange', type_elt, id, val if flagevent
end

#Get a DOM property of an HTML element by Id
def jscript_get_prop(id, prop)
	@suwdlg.execute_script "j6_get_prop ('#{id}', '#{prop}') ; "
	@j_passback
end

#Set a DOM property of an HTML element by Id
def jscript_set_prop(id, prop, svalue)
	svalue = "\"'#{svalue}'\"" if svalue.class == String
	@suwdlg.execute_script "j6_set_prop ('#{id}', '#{prop}', #{svalue}) ; "
	@j_passback
end

def oldjscript_set_prop(id, prop, svalue)
	@suwdlg.execute_script "j6_set_prop ('#{id}', '#{prop}', '#{svalue}') ; "
	@j_passback
end

#Get the attribute of an HTML element by Id
def jscript_get_attr(id, sattr)
	@suwdlg.execute_script "j6_get_attr ('#{id}', '#{sattr}') ; "
	@j_passback
end

#Set a n attribute of an HTML element by Id
def jscript_set_attr(id, sattr, svalue)
	svalue = "\"'#{svalue}'\"" if svalue.class == String
	@suwdlg.execute_script "j6_set_attr ('#{id}', '#{sattr}', #{svalue}) ; "
	@j_passback
end

#Execute a Javscript command
def execute_script(script)
	@suwdlg.execute_script script
end

#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# Private methods to manage HTML and Call Back functions from Jscript
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

#Finalize the HTML String
def assemble_html(html)
	#Top declarations
	text = ""
	text += T_HTML_DOC_TYPE + '<HTML><HEAD>' + T_HTML_SCRIPT_TYPE + T_HTML_UTF_TYPE

	#Styles used in the document
	text += html.get_html_for_all_styles
	
	#Build-in Scripts
	text += built_in_scripts
	
	#Rest of the HEAD Section
	text += html.get_head
	text += "</HEAD>"
	
	#Body section and close of HTML page
	list_events = []
	list_events.push "onload='j6_onload() ;'"
	list_events.push "onunload='j6_onunload() ;'"
	list_events.push "onmousemove='j6_mouse_position() ;'"
	list_events.push "onkeydown='CaptureKeyDown() ;'"
	list_events.push "onkeyup='CaptureKeyUp() ;'"
	list_events.push "onfocus='j6_track_focus() ;'"
	text += "<BODY #{list_events.join(' ')}>"
	
	#Add a space at the top for Mac
	if RUN_ON_MAC
		text += "<br>" 
	end
	
	# Create a hidden field for tracking events on Mac
	if (@asynchronous)
		text += "<input id='HH_SCRIPT_HH' type='hidden' value=''></input>"
	end	
	
	# Inserting the Body part
	text += html.get_body
	
	#Ending the document
	text += "</BODY>"
	text += "</HTML>"
	
	text
end

#provide built-in scripting for the dialog boxes
def built_in_scripts
	text = ""

	text += "<SCRIPT>"
	text += "var $Asynchronous = #{@asynchronous} ;"
	text += "var $NUM_Callback = #{@nb_callback + 1} ;"
	text += "</SCRIPT>"

	text += built_in_script_js
	
	text
end

#Callback to get a values from a java script
def j_getset(p)
	p = decode_param(p.to_s)
	@j_passback = (p == "#nil#") ? nil : p
end

def j_onclose
	unless @closing
		@closing = true
	end	
	@proc_close.call if @proc_close
end

def decode_param(s)
	return nil unless s
	l = s.split(';')
	ln = []
	l.each { |c| ln.push c.to_i }
	svalue = ln.pack("U*")
	svalue
end

#Unique callback dispatch method
def j_dispatch(p=nil)
	if @asynchronous
		val = @suwdlg.get_element_value "HH_SCRIPT_HH"
		val = p unless val && val != ""
	else
		val = p
	end	
	return unless val && val != ""
	lsv = val.split "!CbK!"
	lsv.each do |cbk|
		next if cbk == ""
		lcbk = cbk.split "!ArG!"
		cbk_index = lcbk[0].to_i
		next if (cbk_index <= @nb_callback)
		@nb_callback = cbk_index
		cbk_name = lcbk[1]
		cbk_content = lcbk[2]
		j_callback cbk_name, cbk_content
	end
	if @asynchronous
		@suwdlg.execute_script "j6_clean_callback(#{@nb_callback});"
	end	
end

def j_callback(cbk_name, cbk_content)	
	case cbk_name
	when /Action/i
		return j_action(cbk_content)
	when /GetSet/i
		return j_getset(cbk_content)
	end	
end

#Callback triggered when actions are executed in the web dialog
def j_action(param)
	return unless @hmethod	
	
	#getting the event, type and id of the control
	lsargs = param.split ';;;;'
	event = lsargs[0]
	type = lsargs[1]
	id = lsargs[2]
	svalue = decode_param(lsargs[3])
	onchange = false
	
	
	#Adjusting the event
	case event
	when /wonunload/i
	
	when /wonload/i
		h = svalue.to_i + 50
		@closing = false
		####@suwdlg.set_size @width, h if h < @height && !RUN_ON_MAC
		if h <= @height || (@height_max && h < @height_max)
			@suwdlg.set_size @width, h
		end
		put_focus(@inifocus, @inisel)
		@hmethod.call event, nil, nil, nil
		return
	when /onclick/i
		case type
		when /checkbox/i
			event = "OnChange"
			svalue = (jscript_get_prop(id, "checked")) ? true : false
		end
	when /onchange/i
		onchange = true
		
	when /onkeydown/i

	when /onkeyup/i
	
	else
		return
	end		

	#calling back caller
	val = @hmethod.call event, type, id, svalue
	jscript_set_prop(id, "value", val) if onchange && val != nil
end

end	#class Wdlg

#--------------------------------------------------------------------------------------------------------------
# Class WMsgBox: Utlities to manage Message boxbased on Web Dialog
#--------------------------------------------------------------------------------------------------------------			 

class WMsgBox

#Invoke a modal message box with all possibilities
def WMsgBox.call(message, type=nil, title=nil, lbuttons=["OK"], callback=nil, context=nil)
	wm = WMsgBox.new message, type, title, lbuttons, callback, context
	wm.invoke
end

#Create the class instance
def initialize(message, type=nil, title=nil, lbuttons=["OK"], callback=nil, context=nil)
	@type = type
	@title = title
	@title = "Message" unless @title
	@html_text = message
	@lbuttons = lbuttons
	@callback = callback
	@context = context
	@button_ok = -1
	@button_cancel = -1
end

def invoke
	parse_buttons
	html = compute_html
	@wdlg = Wdlg.new @title
	@wdlg.set_size 500, 140, 300
	@wdlg.set_position 200, 300
	@wdlg.set_html html
	@wdlg.set_background_color 'LightGrey'
	@wdlg.set_callback self.method('visual_callback') 
	@status = nil
	@wdlg.show_modal
	unless @status
		@list_buttons.each_with_index do |lb, i|
			if lb[1] =~ /\^/
				@status = i
				break
			end
		end
		@status = 0 unless @status
	end	
	return @status
end

def visual_callback(event, type, id, svalue)
	case event
	when /onclick/i
		if id =~ /ID__(\d*)/i
			@status = $1.to_i
			@wdlg.close
		end	
	when /onkeydown/i
		lskey = svalue.split '*'
		case lskey[0].to_i
		when 13
			@status = @button_ok
		when 27
			@status = @button_cancel
		else
			return nil
		end
		@wdlg.close	
	end
	
	return nil	
end

#Standard confirmation box for changes
#Return -1 to ignore changes, 0 to cancel and go back, +1 to save changes
def WMsgBox.confirm_changes(txtchanges=nil)
	html_text = T6[:T_WARNING_ActiveChange]
	html_text += "\n<b><span style='color: red ; text-align: center'>" + txtchanges + "</span></b>" if txtchanges
	html_text += "\n" + T6[:T_WARNING_WhatToDo]	
	lbuttons = [:T_BUTTON_Ignore, :T_BUTTON_DecideLater, '^', :T_BUTTON_GoBack, '~', :T_BUTTON_Save]
	title = T6[:T_STR_ConfirmChange]
	UI.beep
	status = WMsgBox.call html_text, nil, title,  T6[lbuttons]
	return ['I', 'L', 'B', 'S'][status]
end

private

def parse_buttons
	@list_buttons = []
	flag = ""
	@lbuttons.each do |b|
		next unless b
		b = b.to_s.strip
		if b =~ /\~\^|\^\~|\~|\^/
			flag += $&
			next
		end	
		@list_buttons.push [b, flag]
		flag = ""
	end	
	@list_buttons[0][1] = '~^' if @list_buttons.length == 1
end

def compute_html
	#initialization
	html = HTML.new
	
	#style used in the dialog box
	html.create_style 'CellImage', nil, 'align: center'
	html.create_style 'CellMessage', nil, 'align: left', 'F-SZ: 10'
	html.create_style 'Button', nil, 'K: black', 'F-SZ: 9'
	html.create_style 'ButtonDef', 'Button', 'border-width: 3px', 'B', 'border-color: green'
	html.create_style 'ButtonCxl', 'Button', 'I', 'B', 'border-width: 2px', 'B', 'border-color: lightpink'

	#formatting the text
	img = MYPLUGIN.picture_get "Button_Add.png"
	txt_img = HTML.format_imagelink img, 48, 48, nil, "CellImage"
	txt = ""
	txt += "<table cellpadding='0' width='100%'><tr>"
	txt += "<td width ='70px'>#{txt_img}</td>"
	txt += "<td><div>"
	txt += HTML.format_para @html_text, nil, 'CellMessage'
	txt += "</div></td></tr></table>"
	
	#formatting the buttons
	nb = @list_buttons.length - 1
	txt_wid = "width='#{(85 / (nb+1)).to_i}%'"
	txt += "<table width='100%'><tr><td width='15%'>&nbsp</td>"
	@list_buttons.each_with_index do |lb, i|
		just = 'center'
		style = "Button"
		if (nb != 0 && i == 0) 
			just = 'left'
		elsif (nb != 0 && i == nb)
			just = 'right'
		end
		if lb[1] =~ /\^/
			style = 'ButtonCxl'
			@button_cancel = i
		end	
		if lb[1] =~ /\~/
			style = 'ButtonDef'
			@button_ok = i
		end	
		txt += "<td #{txt_wid} align='#{just}'>"
		txt += HTML.format_button lb[0], id="ID__#{i}", style
		txt += "</td>"
	end
	txt += "</tr></table>"
	
	#Returning the HTML object
	html.body_add txt
	return html
end

end	#class WMsgBox

#--------------------------------------------------------------------------------------------------------------
# Class SysInfo: Utlities to get some information about the system
#--------------------------------------------------------------------------------------------------------------			 

class SysInfo
@@hsh_prop = nil
#@@asynchronous = (RUBY_PLATFORM =~ /darwin/i) ? true : false	#For Mac asynchronous bug
@@asynchronous = (RUN_ON_MAC) ? true : false	#For Mac asynchronous bug


def SysInfo.[](prop)
	return nil unless prop && prop.strip != ""
	SysInfo.init unless @@hsh_prop
	(@@hsh_prop) ? @@hsh_prop[prop.strip.upcase] : nil
end
	
def SysInfo.init
	return if @@hsh_prop
	text = ""
	text += T_HTML_DOC_TYPE + '<HTML><HEAD>' + T_HTML_SCRIPT_TYPE
	
	txt = %Q~
	function getinfo() {
		Transfer ("UserAgent", navigator.userAgent) ;
		Transfer ("Width", screen.width) ;
		Transfer ("Height", screen.height) ;
		Transfer ("AvailWidth", screen.availWidth) ;
		Transfer ("AvailHeight", screen.availHeight) ;
		Transfer ("AvailTop", screen.availTop) ;
		Transfer ("AvailLeft", screen.availLeft) ;
		Transfer ("ScreenLeft", window.screenTop) ;
		Transfer ("ScreenLeft", window.screenLeft) ;
	}	
	function Transfer(prop, val) {
		msg = '!!' + prop + ';;;;' + val
		obj = document.getElementById ('HH_SCRIPT_HH') ;
		obj.value += msg
		window.location = 'skp:SysInfo@' + msg ;
	}
	~
	
	text += "<SCRIPT>#{txt}</SCRIPT>"	
	text += "</HEAD><BODY onload='getinfo(); '>"
	text += "<input id='HH_SCRIPT_HH' type='hidden' value=''></input>"
	text += "</BODY></HTML>"
	
	unless @@hsh_prop
		@suwdlg = UI::WebDialog.new
		@suwdlg.set_size 10, 10
		@suwdlg.set_html text
		@suwdlg.add_action_callback("SysInfo") { |d, p| SysInfo.callback p }	
		if RUN_ON_MAC
			@suwdlg.show 
			@suwdlg.close
		else
			@suwdlg.show_modal {@suwdlg.close}
		end	
	end	
end

def SysInfo.callback(param)
	if @@asynchronous
		param = @suwdlg.get_element_value "HH_SCRIPT_HH"
	end
	lsv = param.split '!!'
	lsv.each do |p|
		next if p == ""
		SysInfo.store p
	end	
end

def SysInfo.store(p)
	ls = p.split(';;;;')
	code = ls[0]
	val = ls[1]
	
	begin
		case code
		when /userAgent/i
			key = "BROWSER"
			if val =~ /MSIE\s(\d)/i
				value = 'IE' + $1
			elsif val =~ /MSIE/i
				value = 'IE'
			elsif val =~ /Safari/i
				value = (val =~ /version/i) ? value = 'Safari3' : 'Safari2'
			elsif val =~ /Mac/i
				value = 'Safari3'
			else
				value = ''
			end
		else
			key = code
			value = val.to_i
		end
	rescue
		return
	end
	
	@@hsh_prop = {} unless @@hsh_prop
	@@hsh_prop[key] = value
end

end	#class SysInfo

end #Module Traductor
